#Functions and Plots for comparison of models for leukemia risk from benzene exposure
require(ggplot2)

Conversion_from_ppm	= 3.19							#Benzene concentration in ppm can be multiplied by 3.19 to find mg/m3

#All benzene concentrations are written in mg/m3 unless otherwise indicated in comment or in use of "Conversion_from_ppm"ED01	 	= 100.8									#LOA view of concentration of benzene representing a 10% increase in risk for leukemia with 20 year latency
ED01	 	= 19 * Conversion_from_ppm				#LOA view of concentration of benzene representing a 1% increase in absolute risk for acute myeloid leukemia with 20 year latency
LOAEC		= 2 * Conversion_from_ppm				#The lowest observed adverse effect concentration for genetic toxicity
NOAEC		= 0.46 * Conversion_from_ppm			#The no observed adverse effect concentration for genetic toxicity
CompositeAF	= 8										#The assessment factor needed to convert LOAEC to OEL
OEL		 	= 0.25 * Conversion_from_ppm			#The derived no effect level of 0.25 ppm
Upper_Segment_Slope	= 0.1 / ED01					#LOA view of the cancer slope factor for the linear to zero model
Slope_Modification_Factor	= 1 / (LOAEC - OEL)		#Used to find the slope for linear function to reduce cancer slope factor in the CDS model
intercept	= 0 - (Slope_Modification_Factor * OEL)	#Used to find the x-intercept for linear function to reduce cancer slope factor in the CDS model

LNT = function(x , ED01) {							#Linear-to-Origin model
	LNT_slope = 0.01/ED01
	LNT_slope * x}				

CDS = function(x, oel, loaec, ED01, m_slope) {		#Continuously Decreasing Slope model
	LNT_slope = 0.01/ED01
	ifelse(x <= oel, 0,
	       ifelse(( (x > oel) & (x <= loaec)), (m_slope * x + intercept) * LNT_slope * x,
		     LNT_slope * x))}

SMS = function(x, OEL, NOAEC, LOAEC, ED01) {		#Segment Modified Slope model
  #For this model the slope above the LOAEC is increased above the original LNT model
  LNT_slope = 0.01/ED01
  x_one = LOAEC
  x_two = ED01
  y_one = (LNT_slope / 4) * (NOAEC - OEL) + (LNT_slope / 2) * (LOAEC - NOAEC)
  y_two = 0.01
  
  revised_slope = (y_two - y_one) / (x_two - x_one)
  revised_b = 0.01 - (revised_slope * ED01)
    
  ifelse(x <= OEL, 0,
    ifelse( ((x > OEL) & (x <= NOAEC)), (LNT_slope / 4) * (x - OEL),
      ifelse( ((x > NOAEC) & (x <= LOAEC)), (LNT_slope / 4) * (NOAEC - OEL)  +  (LNT_slope / 2) * (x - NOAEC),
       (LNT_slope / 4) * (NOAEC - OEL) + (LNT_slope / 2) * (LOAEC - NOAEC) + revised_slope * (x - LOAEC) ) ) )
}

#Risk calculations for table of commonly used risk values
CDS(4.099, OEL, LOAEC, ED01, Slope_Modification_Factor)	#4 in 10,000
CDS(2.281, OEL, LOAEC, ED01, Slope_Modification_Factor)	#1 in 10,000
CDS(1.6286, OEL, LOAEC, ED01, Slope_Modification_Factor)#4 in 100,000
CDS(1.1039, OEL, LOAEC, ED01, Slope_Modification_Factor)#1 in 100,000
CDS(0.8378, OEL, LOAEC, ED01, Slope_Modification_Factor)#1 in 1,000,000
SMS(5.982, OEL, NOAEC, LOAEC, ED01)						#4 in 10,000
SMS(2.345, OEL, NOAEC, LOAEC, ED01)						#1 in 10,000
SMS(1.6174, OEL, NOAEC, LOAEC, ED01)					#4 in 100,000
SMS(1.04, OEL, NOAEC, LOAEC, ED01)						#1 in 100,000
SMS(0.821, OEL, NOAEC, LOAEC, ED01)						#1 in 1,000,000
LNT(2.42, ED01)											#4 in 10,000
LNT(0.61, ED01)											#1 in 10,000
LNT(0.24, ED01)											#4 in 100,000
LNT(0.061, ED01)										#1 in 100,000
LNT(0.006, ED01)										#1 in 1,000,000

#Risk calculations for table of points of change
CDS(LOAEC, OEL, LOAEC, ED01, Slope_Modification_Factor)	#Risk at LOAEC for CDS model
CDS(NOAEC, OEL, LOAEC, ED01, Slope_Modification_Factor)	#Risk at NOAEC for CDS model
CDS(OEL, OEL, LOAEC, ED01, Slope_Modification_Factor)	#Risk at OEL for CDS model
SMS(LOAEC, OEL, NOAEC, LOAEC, ED01)						#Risk at LOAEC for SMS model
SMS(NOAEC, OEL, NOAEC, LOAEC, ED01)						#Risk at NOAEC for SMS model
SMS(OEL, OEL, NOAEC, LOAEC, ED01)						#Risk at OEL for SMS model
LNT(LOAEC, ED01)										#Risk at LOAEC for LNT model
LNT(NOAEC, ED01)										#Risk at NOAEC for LNT model
LNT(OEL, ED01)											#Risk at OEL for LNT model

#Plot for showing Linear No Threshold model
ggplot(NULL, aes(x=x)) +
  labs(title = "Benzene Acute Myeloid Leukemia Risk in Linear No Threshold Model",
       x = expression(Benzene~Concentration~(mg/m^3)),
       y = "Risk") + 
  stat_function(data = data.frame(x = OEL/10:LOAEC+3), n=10000, fun = LNT, size = 1.5, args = list(ED01 = ED01)) +
  scale_x_continuous(limits = c(0,ED01+3)) +
  scale_y_continuous(limits = c(0, 0.02))
#scale_x_log10(limits = c(OEL/10,LOAEC*10))#breaks = c(OEL/10, OEL, NOAEC, LOAEC), )

#Plot for showing Continuously Decreasing Slope
ggplot(NULL, aes(x=x)) +
		labs(title = "Benzene Acute Myeloid Leukemia Risk in Continously Decreasing Slope Model",
		x = expression(Benzene~Concentration~(mg/m^3)),
		y = "Risk") + 
  stat_function(data = data.frame(x = OEL/10:LOAEC+3), n=10000, fun = CDS, size = 1.5, args = list(oel = OEL, loaec = LOAEC, ED01 = ED01, m_slope = Slope_Modification_Factor)) +
  scale_x_continuous(breaks = c(OEL/10, OEL, NOAEC, LOAEC), limits = c(0,LOAEC+3))
  scale_y_continuous(limits = c(0, 0.015))
  #scale_x_log10(limits = c(OEL/10,LOAEC*10))#breaks = c(OEL/10, OEL, NOAEC, LOAEC), )


#Plot for showing Segmented Modified Slope Model
ggplot(NULL, aes(x=x)) +
  labs(title = "Benzene Acute Myeloid Leukemia Risk in Segmented Modified Slope Model",
       x = expression(Benzene~Concentration~(mg/m^3)),
       y = "Risk") + 
  stat_function(data = data.frame(x = OEL/10:LOAEC+3), n=10000, fun = SMS, size = 1.5, args = list(OEL = OEL, NOAEC = NOAEC, LOAEC = LOAEC, ED01 = ED01)) +
  scale_x_continuous(breaks = c(OEL/10, OEL, NOAEC, LOAEC, ED01), limits = c(0,ED01+3)) +
  scale_y_continuous(limits = c(0, 0.11)) +
  geom_hline(yintercept = 0.1)

  
#Plot for all three functions simultaneously focused on the region around the slope modifications
mainplot = ggplot(NULL, aes(x=x) ) + theme_classic(base_size = 20) +
  theme(plot.title = element_text(hjust = 0.5), legend.key.width = unit(50, "pt")) +
  labs(title = "Models for Benzene \nAcute Myeloid Leukemia Risk",
       x = expression(Benzene~Concentration~(mg/m^3)),
       y = "Risk") + 
  stat_function(data = data.frame(x = 0:LOAEC+1), aes(linetype = "LNT"), n=1000, fun = LNT, size = 1.5, args = list(ED01 = ED01)) +
  stat_function(data = data.frame(x = 0:LOAEC+1), aes(linetype = "CDS"), n=1000, fun = CDS, size = 1.5, args = list(oel = OEL, loaec = LOAEC, ED01 = ED01, m_slope = Slope_Modification_Factor)) +
  stat_function(data = data.frame(x = 0:LOAEC+1), aes(linetype = "SMS"), n=1000, fun = SMS, size = 1.5, args = list(OEL = OEL, NOAEC = NOAEC, LOAEC = LOAEC, ED01 = ED01)) +
  scale_linetype_manual("Models",
	values = c("dashed", "solid", "dotted") ) +
  scale_x_continuous(limits = c(0,LOAEC+1))

  #Save the main plot
  jpeg("Risk Models.jpg", res = 1000, width = 6000, height = 6000, pointsize = 36)
  print(mainplot)
  dev.off()

#A plot focused on the ED01 region
 ED01_plot = ggplot(NULL, aes(x=x) ) + theme_classic(base_size = 20) +
  theme(plot.title = element_text(hjust = 0.5), legend.key.width = unit(50, "pt")) +
  labs(title = "Models for Benzene \nAcute Myeloid Leukemia Risk",
       x = expression(Benzene~Concentration~(mg/m^3)),
       y = "Risk") + 
  stat_function(data = data.frame(x = ED01/3, ED01*3), aes(linetype = "LNT"), n=1000, fun = LNT, size = 1.5, args = list(ED01 = ED01)) +
  stat_function(data = data.frame(x = ED01/3,ED01*3), aes(linetype = "CDS"), fun = CDS, size = 1.5, args = list(oel = OEL, loaec = LOAEC, ED01 = ED01, m_slope = Slope_Modification_Factor)) +
  stat_function(data = data.frame(x = ED01/3,ED01*3), aes(linetype = "SMS"),n=1000, fun = SMS, size = 1.5, args = list(OEL = OEL, NOAEC = NOAEC, LOAEC = LOAEC, ED01 = ED01)) +
  geom_segment(aes(y = 0.01, yend = 0.01, x = ED01/1.5, xend = ED01)) +		#Horizontal line from y-axis to ED01
  geom_segment(aes(y = 0.01, yend = 0.006, x = ED01, xend = ED01)) +		#Drop line from ED01 to x-axis
  scale_x_continuous(limits = c(ED01/1.5,ED01*1.5))

#Plot for all three functions simultaneously for a wide range
wide_view_plot = ggplot(NULL, aes(x=x) ) + theme_classic(base_size = 20) +
  theme(plot.title = element_text(hjust = 0.5), legend.key.width = unit(50, "pt")) +
  labs(title = "Models for Benzene \nAcute Myeloid Leukemia Risk",
       x = expression(Benzene~Concentration~(mg/m^3)),
       y = "Risk") + 
  stat_function(data = data.frame(x = 0:ED01*1.2), aes(linetype = "LNT"), n=1000, fun = LNT, size = 1.5, args = list(ED01 = ED01)) +
  stat_function(data = data.frame(x = 0:ED01*1.2), aes(linetype = "CDS"), n=1000, fun = CDS, size = 1.5, args = list(oel = OEL, loaec = LOAEC, ED01 = ED01, m_slope = Slope_Modification_Factor)) +
  stat_function(data = data.frame(x = 0:ED01*1.2), aes(linetype = "SMS"), n=1000, fun = SMS, size = 1.5, args = list(OEL = OEL, NOAEC = NOAEC, LOAEC = LOAEC, ED01 = ED01)) +
  scale_linetype_manual("Models",
	values = c("dashed", "solid", "dotted") ) +
  scale_x_continuous(limits = c(0,ED01*1.2)) +
  scale_y_continuous(limits = c(0,0.011), breaks = c(0.0025, 0.005, 0.0075, 0.01))

    #Save the wide-view plot
  jpeg("Wide View Risk Models.jpg", res = 1000, width = 6000, height = 6000, pointsize = 36)
  print(wide_view_plot)
  dev.off()